@extends('adminlte::page')

@section('title', 'Edit Petty Cash Expense')

@section('content_header')
<h1>Edit Petty Cash Expense</h1>
@stop

@section('css')
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/tempusdominus-bootstrap-4@5.39.0/build/css/tempusdominus-bootstrap-4.min.css">
@stop

@section('content')

<div class="card">
    <div class="card-body">

        @if($errors->any())
            <div class="alert alert-danger alert-dismissible fade show">
                <ul class="mb-0">
                    @foreach($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
                <button type="button" class="close" data-dismiss="alert">&times;</button>
            </div>
        @endif

        <form method="POST" action="{{ route('admin.pettycash.my-expense.update', $expense->id) }}" enctype="multipart/form-data">
            @csrf
            @method('PUT')


            <div class="form-group">
                <label>Expense Date (DD/MM/YYYY) <span class="text-danger">*</span></label>
                <div class="input-group">
                    <input type="text"
                           name="date_display"
                           id="expense_date_display"
                           class="form-control @error('date') is-invalid @enderror"
                           value="{{ old('date', $expense->expense_date->format('d/m/Y')) }}"
                           placeholder="DD/MM/YYYY"
                           autocomplete="off"
                           required>
                    <div class="input-group-append">
                        <button type="button" class="btn btn-outline-secondary" id="expense-date-picker-btn">
                            <i class="fas fa-calendar-alt"></i>
                        </button>
                    </div>
                </div>
                <input type="hidden" name="date" id="expense_date" value="{{ old('date', $expense->expense_date->format('Y-m-d')) }}">
                @error('date')<span class="invalid-feedback d-block">{{ $message }}</span>@enderror
                <div id="expense-date-error" class="mt-1" style="display: none;">
                    <span class="text-danger"><i class="fas fa-exclamation-circle"></i> <span id="expense-date-error-message"></span></span>
                </div>
                <small class="form-text text-muted">Click calendar icon or enter date in DD/MM/YYYY format. You cannot select future dates.</small>
            </div>

            <div class="form-group">
                <label>Expense Type</label>

                <select name="expense_account_id"
                        class="form-control"
                        required>
                    <option value="">-- Select Expense Type --</option>
                    @foreach($expenseAccounts as $account)
                        <option value="{{ $account->id }}"
                            {{ old('expense_account_id', $expense->expense_account_id) == $account->id ? 'selected' : '' }}>
                            {{ $account->code }} - {{ $account->name }}
                        </option>
                    @endforeach
                </select>
            </div>

            <div class="form-group">
                <label>Description</label>
                <input type="text"
                       name="description"
                       class="form-control"
                       value="{{ old('description', $expense->description) }}"
                       required>
            </div>

            <div class="form-group">
                <label>Amount <span class="text-danger">*</span></label>
                <input type="number"
                       name="amount"
                       id="amount"
                       step="0.01"
                       min="1"
                       class="form-control"
                       value="{{ old('amount', $expense->amount) }}"
                       required>
                <small class="form-text text-muted">Enter the total amount paid (including VAT if applicable)</small>
            </div>

            <div class="form-group">
                <div class="custom-control custom-checkbox">
                    <input type="checkbox"
                           class="custom-control-input"
                           id="is_vat_inclusive"
                           name="is_vat_inclusive"
                           value="1"
                           {{ old('is_vat_inclusive', $expense->is_vat_inclusive) ? 'checked' : '' }}>
                    <label class="custom-control-label" for="is_vat_inclusive">
                        This amount includes 5% VAT
                    </label>
                </div>
                <small class="form-text text-muted">Check this if the amount above includes VAT that you paid on this purchase</small>
            </div>

            <div id="vat_breakdown" style="display: none;">
                <div class="card bg-light">
                    <div class="card-body">
                        <h6 class="card-title">VAT Breakdown</h6>
                        <table class="table table-sm mb-0">
                            <tr>
                                <td><strong>Amount Excluding VAT:</strong></td>
                                <td class="text-right" id="amount_excluding_vat_display">0.00</td>
                            </tr>
                            <tr>
                                <td><strong>VAT (5%):</strong></td>
                                <td class="text-right" id="vat_amount_display">0.00</td>
                            </tr>
                            <tr class="font-weight-bold">
                                <td>Total:</td>
                                <td class="text-right" id="total_amount_display">0.00</td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>

            <input type="hidden" name="vat_amount" id="vat_amount" value="{{ old('vat_amount', $expense->vat_amount) }}">
            <input type="hidden" name="amount_excluding_vat" id="amount_excluding_vat" value="{{ old('amount_excluding_vat', $expense->amount_excluding_vat) }}">

            <div class="form-group">
                <label>Receipt/Invoice Images (Max 3 total)</label>

                @if($expense->receipt_images && count($expense->receipt_images) > 0)
                    <div class="mb-3">
                        <p class="mb-2 text-muted small">Current receipts ({{ count($expense->receipt_images) }}/3):</p>
                        <div class="row">
                            @foreach($expense->receipt_images as $index => $image)
                                <div class="col-md-4 mb-2">
                                    <div class="card">
                                        <div class="card-body p-2">
                                            <a href="{{ asset('storage/' . $image) }}" target="_blank">
                                                <img src="{{ asset('storage/' . $image) }}"
                                                     alt="Receipt {{ $index + 1 }}"
                                                     class="img-fluid img-thumbnail"
                                                     style="max-height: 150px; cursor: pointer; width: 100%; object-fit: cover;">
                                            </a>
                                            <div class="custom-control custom-checkbox mt-2">
                                                <input type="checkbox"
                                                       class="custom-control-input"
                                                       name="delete_images[]"
                                                       value="{{ $image }}"
                                                       id="delete_{{ $index }}">
                                                <label class="custom-control-label text-danger" for="delete_{{ $index }}">
                                                    <small>Delete this image</small>
                                                </label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    </div>
                @endif

                <input type="file"
                       name="receipt_images[]"
                       class="form-control @error('receipt_images.*') is-invalid @error('receipt_images') is-invalid @enderror @enderror"
                       accept="image/*"
                       multiple>
                @error('receipt_images')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
                @error('receipt_images.*')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
                <small class="form-text text-muted">
                    @php
                        $currentCount = $expense->receipt_images ? count($expense->receipt_images) : 0;
                        $remainingSlots = 3 - $currentCount;
                    @endphp
                    @if($remainingSlots > 0)
                        You can upload {{ $remainingSlots }} more image(s).
                    @else
                        You have reached the maximum of 3 images. Delete some to upload new ones.
                    @endif
                    Accepted formats: JPG, PNG, GIF. Max size per image: 1MB.
                </small>
            </div>

            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save"></i> Update Expense
            </button>
            <a href="{{ route('admin.pettycash.my-ledger') }}"
               class="btn btn-secondary">
                Cancel
            </a>

        </form>

    </div>
</div>

@stop

@section('js')
<script src="https://cdn.jsdelivr.net/npm/moment@2.29.4/moment.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/tempusdominus-bootstrap-4@5.39.0/build/js/tempusdominus-bootstrap-4.min.js"></script>
<script>
    // Function to parse DD/MM/YYYY to Date object
    function parseDate(dateString) {
        const parts = dateString.split('/');
        if (parts.length !== 3) return null;

        const day = parseInt(parts[0], 10);
        const month = parseInt(parts[1], 10) - 1; // Month is 0-indexed
        const year = parseInt(parts[2], 10);

        if (isNaN(day) || isNaN(month) || isNaN(year)) return null;
        if (day < 1 || day > 31 || month < 0 || month > 11 || year < 1900 || year > 2100) return null;

        return new Date(year, month, day);
    }

    // Function to format Date to YYYY-MM-DD
    function formatDateForServer(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    // Function to validate expense date
    function validateExpenseDate(dateString) {
        const errorDiv = document.getElementById('expense-date-error');
        const errorMessage = document.getElementById('expense-date-error-message');
        const submitButton = document.querySelector('button[type="submit"]');
        const dateInput = document.getElementById('expense_date');

        // Clear previous errors
        errorDiv.style.display = 'none';
        errorMessage.textContent = '';

        // Parse date
        const selectedDate = parseDate(dateString);

        if (!selectedDate || isNaN(selectedDate.getTime())) {
            errorMessage.textContent = 'Invalid date format. Please use DD/MM/YYYY.';
            errorDiv.style.display = 'block';
            submitButton.disabled = true;
            return false;
        }

        // Check if date is in the future
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        selectedDate.setHours(0, 0, 0, 0);

        if (selectedDate > today) {
            errorMessage.textContent = 'You cannot add expenses for future dates. Please select today or a past date.';
            errorDiv.style.display = 'block';
            submitButton.disabled = true;
            return false;
        }

        // Update hidden input with server format
        dateInput.value = formatDateForServer(selectedDate);
        submitButton.disabled = false;

        return true;
    }

    document.addEventListener('DOMContentLoaded', function() {
        const dateDisplayInput = document.getElementById('expense_date_display');
        const submitButton = document.querySelector('button[type="submit"]');
        const form = document.querySelector('form');

        // Initialize datepicker
        $('#expense_date_display').datetimepicker({
            format: 'DD/MM/YYYY',
            maxDate: moment(),  // Prevent future dates
            useCurrent: false,
            icons: {
                time: 'far fa-clock',
                date: 'far fa-calendar',
                up: 'fas fa-arrow-up',
                down: 'fas fa-arrow-down',
                previous: 'fas fa-chevron-left',
                next: 'fas fa-chevron-right',
                today: 'fas fa-calendar-check',
                clear: 'far fa-trash-alt',
                close: 'far fa-times-circle'
            }
        });

        // When date is selected from datepicker, validate it
        $('#expense_date_display').on('change.datetimepicker', function(e) {
            if (e.date) {
                const formattedDate = e.date.format('DD/MM/YYYY');
                dateDisplayInput.value = formattedDate;
                validateExpenseDate(formattedDate);
            }
        });

        // Trigger datepicker when calendar button is clicked
        $('#expense-date-picker-btn').on('click', function() {
            $('#expense_date_display').datetimepicker('show');
        });

        // Auto-format date as user types
        dateDisplayInput.addEventListener('input', function(e) {
            let value = this.value.replace(/\D/g, ''); // Remove non-digits
            let formattedValue = '';

            if (value.length > 0) {
                // Add day (first 2 digits)
                formattedValue = value.substring(0, 2);

                // Add slash after day if we have 2 or more digits
                if (value.length >= 2) {
                    formattedValue += '/';
                }

                if (value.length >= 3) {
                    // Add month (next 2 digits)
                    formattedValue += value.substring(2, 4);
                }

                // Add slash after month if we have 4 or more digits
                if (value.length >= 4) {
                    formattedValue += '/';
                }

                if (value.length >= 5) {
                    // Add year (next 4 digits)
                    formattedValue += value.substring(4, 8);
                }
            }

            this.value = formattedValue;
            validateExpenseDate(this.value);
        });

        dateDisplayInput.addEventListener('blur', function(e) {
            validateExpenseDate(this.value);
        });

        // Validate on page load
        if (dateDisplayInput.value) {
            validateExpenseDate(dateDisplayInput.value);
        }

        // Prevent form submission if date is invalid
        form.addEventListener('submit', function(e) {
            if (submitButton.disabled) {
                e.preventDefault();
                alert('Please fix the date errors before submitting.');
                return false;
            }
        });

        // VAT calculation functionality
        const amountInput = document.getElementById('amount');
        const vatCheckbox = document.getElementById('is_vat_inclusive');
        const vatBreakdown = document.getElementById('vat_breakdown');
        const vatAmountHidden = document.getElementById('vat_amount');
        const amountExcludingVatHidden = document.getElementById('amount_excluding_vat');

        function calculateVAT() {
            const totalAmount = parseFloat(amountInput.value) || 0;
            const isVatInclusive = vatCheckbox.checked;

            if (isVatInclusive && totalAmount > 0) {
                // Calculate VAT from inclusive amount
                // Total = Amount + VAT
                // Total = Amount + (Amount * 0.05)
                // Total = Amount * 1.05
                // Amount = Total / 1.05
                const amountExcludingVat = totalAmount / 1.05;
                const vatAmount = totalAmount - amountExcludingVat;

                // Update display
                document.getElementById('amount_excluding_vat_display').textContent = amountExcludingVat.toFixed(2);
                document.getElementById('vat_amount_display').textContent = vatAmount.toFixed(2);
                document.getElementById('total_amount_display').textContent = totalAmount.toFixed(2);

                // Update hidden inputs
                vatAmountHidden.value = vatAmount.toFixed(2);
                amountExcludingVatHidden.value = amountExcludingVat.toFixed(2);

                // Show breakdown
                vatBreakdown.style.display = 'block';
            } else {
                // Hide breakdown and reset values
                vatBreakdown.style.display = 'none';
                vatAmountHidden.value = '0';
                amountExcludingVatHidden.value = '0';
            }
        }

        // Event listeners for VAT calculation
        amountInput.addEventListener('input', calculateVAT);
        vatCheckbox.addEventListener('change', calculateVAT);

        // Initial calculation on page load
        calculateVAT();
    });
</script>
@stop
