@extends('adminlte::page')

@section('title', 'Mark Attendance')

@section('content_header')
    <h1>Mark Attendance</h1>
@stop

@section('css')
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/tempusdominus-bootstrap-4@5.39.0/build/css/tempusdominus-bootstrap-4.min.css">
    <style>
        #submit-button:disabled,
        #submit-button.disabled {
            opacity: 0.5 !important;
            cursor: not-allowed !important;
            pointer-events: none !important;
            background-color: #6c757d !important;
            border-color: #6c757d !important;
        }
    </style>
@stop

@section('content')
    <div class="row">
        <div class="col-md-8 offset-md-2">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Attendance for {{ $employee->full_name }}</h3>
                </div>
                <form action="{{ route('admin.hr.attendance.store') }}" method="POST">
                    @csrf
                    <div class="card-body">
                        @if($todayAttendance)
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle"></i> You have already marked your attendance for today. You can update it below.
                            </div>
                        @endif

                        <div class="form-group">
                            <label for="date">Date (DD/MM/YYYY) <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <input type="text"
                                       name="date_display"
                                       id="date_display"
                                       class="form-control @error('date') is-invalid @enderror"
                                       value="{{ old('date', $todayAttendance ? $todayAttendance->date->format('d/m/Y') : now()->format('d/m/Y')) }}"
                                       placeholder="DD/MM/YYYY"
                                       autocomplete="off"
                                       required>
                                <div class="input-group-append">
                                    <button type="button" class="btn btn-outline-secondary" id="date-picker-btn">
                                        <i class="fas fa-calendar-alt"></i>
                                    </button>
                                </div>
                            </div>
                            <input type="hidden" name="date" id="date" value="{{ old('date', $todayAttendance ? $todayAttendance->date->format('Y-m-d') : now()->format('Y-m-d')) }}">
                            @error('date')<span class="invalid-feedback d-block">{{ $message }}</span>@enderror
                            <div id="date-error" class="mt-1" style="display: none;">
                                <span class="text-danger"><i class="fas fa-exclamation-circle"></i> <span id="date-error-message"></span></span>
                            </div>
                            <small class="form-text text-muted">Click calendar icon or enter date in DD/MM/YYYY format</small>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="check_in">Check In Time <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <input type="time"
                                               name="check_in"
                                               id="check_in"
                                               class="form-control @error('check_in') is-invalid @enderror"
                                               value="{{ old('check_in', $todayAttendance ? substr($todayAttendance->check_in, 0, 5) : '08:00') }}"
                                               step="1800"
                                               readonly
                                               required>
                                        <div class="input-group-append">
                                            <button type="button" class="btn btn-outline-secondary time-up" data-target="check_in">
                                                <i class="fas fa-chevron-up"></i>
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary time-down" data-target="check_in">
                                                <i class="fas fa-chevron-down"></i>
                                            </button>
                                        </div>
                                    </div>
                                    @error('check_in')<span class="invalid-feedback d-block">{{ $message }}</span>@enderror
                                    <small class="form-text text-muted">Default: 8:00 AM. Click arrows for 30-min increments.</small>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="check_out">Check Out Time</label>
                                    <div class="input-group">
                                        <input type="time"
                                               name="check_out"
                                               id="check_out"
                                               class="form-control @error('check_out') is-invalid @enderror"
                                               value="{{ old('check_out', $todayAttendance && $todayAttendance->check_out ? substr($todayAttendance->check_out, 0, 5) : '18:00') }}"
                                               step="1800"
                                               readonly>
                                        <div class="input-group-append">
                                            <button type="button" class="btn btn-outline-secondary time-up" data-target="check_out">
                                                <i class="fas fa-chevron-up"></i>
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary time-down" data-target="check_out">
                                                <i class="fas fa-chevron-down"></i>
                                            </button>
                                        </div>
                                    </div>
                                    @error('check_out')<span class="invalid-feedback d-block">{{ $message }}</span>@enderror
                                    <small class="form-text text-muted">Default: 6:00 PM. Click arrows for 30-min increments.</small>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="break_start">Break Start</label>
                                    <div class="input-group">
                                        <input type="time"
                                               name="break_start"
                                               id="break_start"
                                               class="form-control @error('break_start') is-invalid @enderror"
                                               value="{{ old('break_start', $todayAttendance && $todayAttendance->break_start ? substr($todayAttendance->break_start, 0, 5) : '12:00') }}"
                                               step="1800"
                                               readonly>
                                        <div class="input-group-append">
                                            <button type="button" class="btn btn-outline-secondary time-up" data-target="break_start">
                                                <i class="fas fa-chevron-up"></i>
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary time-down" data-target="break_start">
                                                <i class="fas fa-chevron-down"></i>
                                            </button>
                                        </div>
                                    </div>
                                    @error('break_start')<span class="invalid-feedback d-block">{{ $message }}</span>@enderror
                                    <small class="form-text text-muted">Default: 12:00 PM. Click arrows for 30-min increments.</small>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="break_end">Break End</label>
                                    <div class="input-group">
                                        <input type="time"
                                               name="break_end"
                                               id="break_end"
                                               class="form-control @error('break_end') is-invalid @enderror"
                                               value="{{ old('break_end', $todayAttendance && $todayAttendance->break_end ? substr($todayAttendance->break_end, 0, 5) : '13:00') }}"
                                               step="1800"
                                               readonly>
                                        <div class="input-group-append">
                                            <button type="button" class="btn btn-outline-secondary time-up" data-target="break_end">
                                                <i class="fas fa-chevron-up"></i>
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary time-down" data-target="break_end">
                                                <i class="fas fa-chevron-down"></i>
                                            </button>
                                        </div>
                                    </div>
                                    @error('break_end')<span class="invalid-feedback d-block">{{ $message }}</span>@enderror
                                    <small class="form-text text-muted">Default: 1:00 PM. Click arrows for 30-min increments.</small>
                                </div>
                            </div>
                        </div>

                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i>
                            <strong>Note:</strong> If check-out time is past 12:00 AM (midnight), it will be considered as next day (for late-night shifts).
                        </div>

                        <div class="form-group">
                            <label for="status">Status <span class="text-danger">*</span></label>
                            <select name="status" id="status" class="form-control @error('status') is-invalid @enderror" required>
                                <option value="present" {{ old('status', $todayAttendance->status ?? 'present') == 'present' ? 'selected' : '' }}>Present</option>
                                <option value="late" {{ old('status', $todayAttendance->status ?? '') == 'late' ? 'selected' : '' }}>Late</option>
                                <option value="half_day" {{ old('status', $todayAttendance->status ?? '') == 'half_day' ? 'selected' : '' }}>Half Day</option>
                                <option value="absent" {{ old('status', $todayAttendance->status ?? '') == 'absent' ? 'selected' : '' }}>Absent</option>
                                <option value="on_leave" {{ old('status', $todayAttendance->status ?? '') == 'on_leave' ? 'selected' : '' }}>On Leave</option>
                            </select>
                            @error('status')<span class="invalid-feedback">{{ $message }}</span>@enderror
                        </div>

                        <div class="form-group">
                            <label for="notes">Notes</label>
                            <textarea name="notes" id="notes" class="form-control @error('notes') is-invalid @enderror" rows="3">{{ old('notes', $todayAttendance->notes ?? '') }}</textarea>
                            @error('notes')<span class="invalid-feedback">{{ $message }}</span>@enderror
                        </div>
                    </div>
                    <div class="card-footer">
                        <button type="submit" class="btn btn-primary" id="submit-button">
                            <i class="fas fa-check"></i> {{ $todayAttendance ? 'Update' : 'Mark' }} Attendance
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
@stop

@section('js')
<script src="https://cdn.jsdelivr.net/npm/moment@2.29.4/moment.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/tempusdominus-bootstrap-4@5.39.0/build/js/tempusdominus-bootstrap-4.min.js"></script>
<script>
    // Function to adjust time by 30 minutes
    function adjustTime(inputId, direction) {
        const input = document.getElementById(inputId);
        const currentValue = input.value || '00:00';
        const [hours, minutes] = currentValue.split(':').map(Number);

        // Convert to total minutes
        let totalMinutes = hours * 60 + minutes;

        // Adjust by 30 minutes
        if (direction === 'up') {
            totalMinutes += 30;
        } else {
            totalMinutes -= 30;
        }

        // Handle overflow/underflow (wrap around 24 hours)
        if (totalMinutes < 0) {
            totalMinutes = 1440 + totalMinutes; // 1440 = 24 * 60
        } else if (totalMinutes >= 1440) {
            totalMinutes = totalMinutes - 1440;
        }

        // Convert back to hours and minutes
        const newHours = Math.floor(totalMinutes / 60);
        const newMinutes = totalMinutes % 60;

        // Format as HH:MM
        const formattedTime = String(newHours).padStart(2, '0') + ':' + String(newMinutes).padStart(2, '0');
        input.value = formattedTime;
    }

    // Function to parse DD/MM/YYYY to Date object
    function parseDate(dateString) {
        const parts = dateString.split('/');
        if (parts.length !== 3) return null;

        const day = parseInt(parts[0], 10);
        const month = parseInt(parts[1], 10) - 1; // Month is 0-indexed
        const year = parseInt(parts[2], 10);

        if (isNaN(day) || isNaN(month) || isNaN(year)) return null;
        if (day < 1 || day > 31 || month < 0 || month > 11 || year < 1900 || year > 2100) return null;

        return new Date(year, month, day);
    }

    // Function to format Date to YYYY-MM-DD
    function formatDateForServer(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    // Function to validate date
    function validateDate(dateString) {
        const errorDiv = document.getElementById('date-error');
        const errorMessage = document.getElementById('date-error-message');
        const submitButton = document.getElementById('submit-button');
        const dateInput = document.getElementById('date');

        // Clear previous errors
        errorDiv.style.display = 'none';
        errorMessage.textContent = '';

        // Parse date
        const selectedDate = parseDate(dateString);

        if (!selectedDate || isNaN(selectedDate.getTime())) {
            errorMessage.textContent = 'Invalid date format. Please use DD/MM/YYYY.';
            errorDiv.style.display = 'block';
            submitButton.disabled = true;
            return false;
        }

        // Check if date is in the future
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        selectedDate.setHours(0, 0, 0, 0);

        if (selectedDate > today) {
            errorMessage.textContent = 'You cannot mark attendance for future dates. Please select today or a past date.';
            errorDiv.style.display = 'block';
            submitButton.disabled = true;
            return false;
        }

        // Update hidden input with server format
        dateInput.value = formatDateForServer(selectedDate);

        // Temporarily disable button while checking
        submitButton.disabled = true;

        // Check if attendance already exists via AJAX
        checkAttendanceExists(formatDateForServer(selectedDate));

        return true;
    }

    // Function to check if attendance already exists
    function checkAttendanceExists(dateString) {
        const errorDiv = document.getElementById('date-error');
        const errorMessage = document.getElementById('date-error-message');
        const submitButton = document.getElementById('submit-button');

        console.log('Checking attendance for date:', dateString);

        fetch(`{{ route('admin.hr.attendance.check') }}?date=${dateString}`, {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            console.log('Attendance check response:', data);

            if (data.exists && !data.is_today) {
                // Attendance exists for a past date - disable button completely
                console.log('Disabling button - attendance exists for past date');
                errorMessage.textContent = 'You have already marked attendance for this date. Please choose a different date.';
                errorDiv.style.display = 'block';
                submitButton.disabled = true;
                submitButton.classList.add('disabled');
                submitButton.setAttribute('disabled', 'disabled');
                // Keep original button text or show as disabled
                submitButton.innerHTML = '<i class="fas fa-ban"></i> Already Marked';
                console.log('Button disabled state:', submitButton.disabled);
                console.log('Button has disabled class:', submitButton.classList.contains('disabled'));
            } else if (data.exists && data.is_today) {
                // Allow updating today's attendance
                console.log('Enabling button - updating today\'s attendance');
                errorDiv.style.display = 'none';
                submitButton.disabled = false;
                submitButton.classList.remove('disabled');
                submitButton.removeAttribute('disabled');
                // Update button text to "Update Attendance"
                submitButton.innerHTML = '<i class="fas fa-check"></i> Update Attendance';
            } else {
                // No attendance exists - allow marking new attendance
                console.log('Enabling button - marking new attendance');
                errorDiv.style.display = 'none';
                submitButton.disabled = false;
                submitButton.classList.remove('disabled');
                submitButton.removeAttribute('disabled');
                // Update button text to "Mark Attendance"
                submitButton.innerHTML = '<i class="fas fa-check"></i> Mark Attendance';
            }
        })
        .catch(error => {
            console.error('Error checking attendance:', error);
            submitButton.disabled = false;
            submitButton.classList.remove('disabled');
        });
    }

    // Add event listeners to all time adjustment buttons
    document.addEventListener('DOMContentLoaded', function() {
        const dateDisplayInput = document.getElementById('date_display');
        const submitButton = document.getElementById('submit-button');
        const form = document.querySelector('form');

        // Initialize datepicker
        $('#date_display').datetimepicker({
            format: 'DD/MM/YYYY',
            maxDate: moment(),  // Prevent future dates
            useCurrent: false,
            icons: {
                time: 'far fa-clock',
                date: 'far fa-calendar',
                up: 'fas fa-arrow-up',
                down: 'fas fa-arrow-down',
                previous: 'fas fa-chevron-left',
                next: 'fas fa-chevron-right',
                today: 'fas fa-calendar-check',
                clear: 'far fa-trash-alt',
                close: 'far fa-times-circle'
            }
        });

        // When date is selected from datepicker, validate it
        $('#date_display').on('change.datetimepicker', function(e) {
            if (e.date) {
                const formattedDate = e.date.format('DD/MM/YYYY');
                dateDisplayInput.value = formattedDate;
                validateDate(formattedDate);
            }
        });

        // Trigger datepicker when calendar button is clicked
        $('#date-picker-btn').on('click', function() {
            $('#date_display').datetimepicker('show');
        });

        // Auto-format date as user types
        dateDisplayInput.addEventListener('input', function(e) {
            let value = this.value.replace(/\D/g, ''); // Remove non-digits
            let formattedValue = '';

            if (value.length > 0) {
                // Add day (first 2 digits)
                formattedValue = value.substring(0, 2);

                // Add slash after day if we have 2 or more digits
                if (value.length >= 2) {
                    formattedValue += '/';
                }

                if (value.length >= 3) {
                    // Add month (next 2 digits)
                    formattedValue += value.substring(2, 4);
                }

                // Add slash after month if we have 4 or more digits
                if (value.length >= 4) {
                    formattedValue += '/';
                }

                if (value.length >= 5) {
                    // Add year (next 4 digits)
                    formattedValue += value.substring(4, 8);
                }
            }

            this.value = formattedValue;
            validateDate(this.value);
        });

        dateDisplayInput.addEventListener('blur', function(e) {
            validateDate(this.value);
        });

        // Validate on page load
        if (dateDisplayInput.value) {
            validateDate(dateDisplayInput.value);
        }

        // Prevent form submission if date is invalid or button is disabled
        form.addEventListener('submit', function(e) {
            if (submitButton.disabled || submitButton.classList.contains('disabled')) {
                e.preventDefault();
                e.stopPropagation();
                alert('Please fix the date errors before submitting.');
                return false;
            }
        });

        // Additional click handler on submit button to prevent any action when disabled
        submitButton.addEventListener('click', function(e) {
            if (this.disabled || this.classList.contains('disabled')) {
                e.preventDefault();
                e.stopPropagation();
                e.stopImmediatePropagation();
                return false;
            }
        });

        // Up buttons
        document.querySelectorAll('.time-up').forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const targetId = this.getAttribute('data-target');
                adjustTime(targetId, 'up');
            });
        });

        // Down buttons
        document.querySelectorAll('.time-down').forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const targetId = this.getAttribute('data-target');
                adjustTime(targetId, 'down');
            });
        });

        // Handle status change to disable/enable time fields
        const statusSelect = document.getElementById('status');
        const checkInInput = document.getElementById('check_in');
        const checkOutInput = document.getElementById('check_out');
        const breakStartInput = document.getElementById('break_start');
        const breakEndInput = document.getElementById('break_end');
        const timeButtons = document.querySelectorAll('.time-up, .time-down');

        function handleStatusChange() {
            const status = statusSelect.value;

            if (status === 'absent' || status === 'on_leave') {
                // Set all time fields to 00:00
                checkInInput.value = '00:00';
                checkOutInput.value = '00:00';
                breakStartInput.value = '00:00';
                breakEndInput.value = '00:00';

                // Disable all time fields and buttons
                checkInInput.disabled = true;
                checkOutInput.disabled = true;
                breakStartInput.disabled = true;
                breakEndInput.disabled = true;
                timeButtons.forEach(btn => btn.disabled = true);
            } else {
                // Enable time fields and buttons
                checkInInput.disabled = false;
                checkOutInput.disabled = false;
                breakStartInput.disabled = false;
                breakEndInput.disabled = false;
                timeButtons.forEach(btn => btn.disabled = false);

                // Reset to default values if currently 00:00
                if (checkInInput.value === '00:00') {
                    checkInInput.value = '08:00';
                }
                if (checkOutInput.value === '00:00') {
                    checkOutInput.value = '18:00';
                }
                if (breakStartInput.value === '00:00') {
                    breakStartInput.value = '12:00';
                }
                if (breakEndInput.value === '00:00') {
                    breakEndInput.value = '13:00';
                }
            }
        }

        // Listen for status changes
        statusSelect.addEventListener('change', handleStatusChange);

        // Run on page load to handle existing status
        handleStatusChange();
    });
</script>
@stop
