/**
 * Custom JavaScript for RMS Application
 *
 * This file contains global JavaScript functionality that applies
 * across the entire application.
 */

/**
 * Prevent Double Form Submission
 *
 * Automatically disables submit buttons after form submission
 * to prevent accidental duplicate submissions.
 */
document.addEventListener('DOMContentLoaded', function() {
    console.log('Custom JS: Form submission prevention initialized');

    // Get all forms on the page
    const forms = document.querySelectorAll('form');

    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            // Check if form has novalidate attribute (skip for these forms)
            if (this.hasAttribute('data-no-disable')) {
                return;
            }

            // Find all submit buttons in this form
            const submitButtons = this.querySelectorAll('button[type="submit"], input[type="submit"]');

            // Disable all submit buttons
            submitButtons.forEach(button => {
                // Don't disable if button already has data-no-disable attribute
                if (button.hasAttribute('data-no-disable')) {
                    return;
                }

                button.disabled = true;

                // Add visual feedback for buttons
                if (button.tagName === 'BUTTON') {
                    const originalHTML = button.innerHTML;
                    button.setAttribute('data-original-html', originalHTML);

                    // Add spinner and "Processing..." text
                    button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';

                    // Add Bootstrap disabled styling
                    button.classList.add('disabled');
                }
            });

            // Re-enable buttons after 5 seconds as a safety measure
            // (in case form validation fails or navigation doesn't happen)
            setTimeout(() => {
                submitButtons.forEach(button => {
                    button.disabled = false;
                    button.classList.remove('disabled');

                    if (button.tagName === 'BUTTON' && button.hasAttribute('data-original-html')) {
                        button.innerHTML = button.getAttribute('data-original-html');
                        button.removeAttribute('data-original-html');
                    }
                });
            }, 5000);
        });
    });
});

/**
 * Auto-hide alert messages after 5 seconds
 */
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');

    alerts.forEach(alert => {
        // Don't auto-hide error alerts
        if (alert.classList.contains('alert-danger') || alert.classList.contains('alert-error')) {
            return;
        }

        setTimeout(() => {
            // Fade out effect
            alert.style.transition = 'opacity 0.5s ease';
            alert.style.opacity = '0';

            setTimeout(() => {
                alert.remove();
            }, 500);
        }, 5000);
    });
});
