<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Employee;
use App\Models\Attendance;
use App\Models\Account;
use App\Models\PettyCashExpense;
use Carbon\Carbon;

class UserController extends Controller
{
    public function dashboard()
    {
        $user = Auth::user();

        // Get employee record
        $employee = Employee::where('user_id', $user->id)->with(['department', 'designation'])->first();

        // Attendance Summary (current month)
        $attendanceStats = null;
        if ($employee) {
            $currentMonth = Carbon::now()->startOfMonth();
            $endOfMonth = Carbon::now()->endOfMonth();

            $attendances = Attendance::where('employee_id', $employee->id)
                ->whereBetween('date', [$currentMonth, $endOfMonth])
                ->get();

            $daysPresent = $attendances->whereIn('status', ['present', 'late', 'half_day'])->count();
            $daysAbsent = $attendances->where('status', 'absent')->count();
            $daysOnLeave = $attendances->where('status', 'on_leave')->count();
            $totalOT = 0;
            foreach ($attendances->whereNotNull('total_hours') as $att) {
                $totalOT += ($att->total_hours - 9);
            }

            $attendanceStats = [
                'days_present' => $daysPresent,
                'days_absent' => $daysAbsent,
                'days_on_leave' => $daysOnLeave,
                'total_ot' => $totalOT,
                'pending_approvals' => $attendances->whereNull('approved_by')->count(),
            ];
        }

        // Petty Cash Summary
        $pettyCashStats = null;
        $pettyCashAccount = Account::where('user_id', $user->id)
            ->where('is_posting', 1)
            ->whereHas('parent', function ($query) {
                $query->where('name', 'Petty Cash');
            })
            ->first();

        if ($pettyCashAccount) {
            $pendingExpenses = PettyCashExpense::where('user_id', $user->id)
                ->where('petty_cash_account_id', $pettyCashAccount->id)
                ->where('status', 'pending')
                ->count();

            $approvedThisMonth = PettyCashExpense::where('user_id', $user->id)
                ->where('petty_cash_account_id', $pettyCashAccount->id)
                ->where('status', 'approved')
                ->whereMonth('expense_date', Carbon::now()->month)
                ->whereYear('expense_date', Carbon::now()->year)
                ->sum('amount');

            $pettyCashStats = [
                'balance' => $pettyCashAccount->balance,
                'account_name' => $pettyCashAccount->name,
                'pending_expenses' => $pendingExpenses,
                'approved_this_month' => $approvedThisMonth,
            ];
        }

        return view('user.dashboard', compact('user', 'employee', 'attendanceStats', 'pettyCashStats'));
    }
}

