<?php

namespace App\Http\Controllers\Admin\HR;

use App\Http\Controllers\Controller;
use App\Models\LeaveApplication;
use App\Models\LeaveType;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class LeaveApplicationController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $employee = Employee::where('user_id', $user->id)->first();

        if (!$employee) {
            return redirect()->route('admin.dashboard')
                ->with('error', 'You are not linked to an employee record.');
        }

        $leaves = LeaveApplication::with(['leaveType', 'managerApprover', 'hrApprover'])
            ->where('employee_id', $employee->id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('admin.hr.leaves.index', compact('leaves'));
    }

    public function all(Request $request)
    {
        $query = LeaveApplication::with(['employee.department', 'employee.designation', 'leaveType', 'managerApprover', 'hrApprover']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by employee
        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->employee_id);
        }

        // Filter by date range
        if ($request->filled('start_date')) {
            $query->where('start_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('end_date', '<=', $request->end_date);
        }

        $leaves = $query->orderBy('created_at', 'desc')->paginate(15);
        $employees = Employee::where('status', 'active')->get();

        return view('admin.hr.leaves.all', compact('leaves', 'employees'));
    }

    public function create()
    {
        $user = Auth::user();
        $employee = Employee::where('user_id', $user->id)->first();

        if (!$employee) {
            return redirect()->route('admin.dashboard')
                ->with('error', 'You are not linked to an employee record.');
        }

        $leaveTypes = LeaveType::where('is_active', true)->get();

        return view('admin.hr.leaves.create', compact('leaveTypes', 'employee'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        $employee = Employee::where('user_id', $user->id)->first();

        if (!$employee) {
            return redirect()->route('admin.dashboard')
                ->with('error', 'You are not linked to an employee record.');
        }

        $validated = $request->validate([
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required|string',
        ]);

        // Calculate total days
        $startDate = Carbon::parse($validated['start_date']);
        $endDate = Carbon::parse($validated['end_date']);
        $totalDays = $startDate->diffInDays($endDate) + 1;

        $validated['employee_id'] = $employee->id;
        $validated['total_days'] = $totalDays;
        $validated['status'] = 'pending';

        LeaveApplication::create($validated);

        return redirect()->route('admin.hr.leaves.index')
            ->with('success', 'Leave application submitted successfully.');
    }

    public function show(LeaveApplication $leave)
    {
        $leave->load(['employee.department', 'employee.designation', 'leaveType', 'managerApprover', 'hrApprover']);

        return view('admin.hr.leaves.show', compact('leave'));
    }

    public function managerApprove(Request $request, LeaveApplication $leave)
    {
        if ($leave->status !== 'pending') {
            return redirect()->back()
                ->with('error', 'This leave application cannot be approved.');
        }

        $validated = $request->validate([
            'manager_comments' => 'nullable|string',
        ]);

        $leave->update([
            'status' => 'manager_approved',
            'manager_approved_by' => Auth::id(),
            'manager_approved_at' => now(),
            'manager_comments' => $validated['manager_comments'] ?? null,
        ]);

        return redirect()->back()
            ->with('success', 'Leave application approved by manager. Awaiting HR approval.');
    }

    public function hrApprove(Request $request, LeaveApplication $leave)
    {
        if ($leave->status !== 'manager_approved') {
            return redirect()->back()
                ->with('error', 'This leave application requires manager approval first.');
        }

        $validated = $request->validate([
            'hr_comments' => 'nullable|string',
        ]);

        $leave->update([
            'status' => 'hr_approved',
            'hr_approved_by' => Auth::id(),
            'hr_approved_at' => now(),
            'hr_comments' => $validated['hr_comments'] ?? null,
        ]);

        return redirect()->back()
            ->with('success', 'Leave application fully approved.');
    }

    public function reject(Request $request, LeaveApplication $leave)
    {
        if ($leave->status === 'hr_approved') {
            return redirect()->back()
                ->with('error', 'Cannot reject an already approved leave application.');
        }

        $validated = $request->validate([
            'rejection_reason' => 'required|string',
        ]);

        $leave->update([
            'status' => 'rejected',
            'rejection_reason' => $validated['rejection_reason'],
        ]);

        return redirect()->back()
            ->with('success', 'Leave application rejected.');
    }
}
